;****************************************************************
;*								*
;*	Program	: Routine to imitate Apricot Bios 3 serial	*
;*		  device interface to MS-BASIC using the	*
;*		  IBM BIOS interrupt 14h interface		*
;*								*
;*	Author 	: J.Edge					*
;*								*
;*	Version	: 1.0 (beta)					*
;*								*
;*	Date	:	Original version 05/04/89		*
;*			This version     05/04/89		*
;*								*
;****************************************************************
;*								*
;*	Usage	: CALL IOINT(DEV%,COM%,DAT%,RET%)		*
;*		  See Apricot BIOS 3 Tech. Ref manual for	*
;*		  further details				*
;*								*
;*	Restrictions:						*
;*		  Calls 13h and above, and calls A,B,E and	*
;*		  F are not supported by this interface		*
;*		  though no damage will be caused by using	*
;*		  them						*
;*								*
;*		  Split baud rates, data bits			*
;*		  are not supported. i.e. Transmit and		*
;*		  receive values for these items must be	*
;*		  the same (this can be fixed)			*
;*								*
;*		  Certain baud rates are not supported		*
;*		  (see S_ConfTxBaud for details)		*
;*		  (this can be fixed)				*
;*								*
;*		  Only 7 or 8 data bits may be used		*
;*								*
;*		  Only 1 or 2 stop bits may be used		*
;*								*
;****************************************************************


;********************** EQUATES

SERIALDEV	equ	34h	; Serial device number in apricot interface
PORTNUM		equ	0	; Serial port number (0=system board)
MODEMCPORT	equ	3FCh	; Modem control port
DTRLINE		equ	00000001B	; DTR and mask for modem control port
RTSLINE		equ	00000010B	; RTS and mask for modem contol port
CTSLINE		equ	00010000B	; CTS and mask for modem line port
DSRLINE		equ	00100000B	; DSR and mask for modem line port
DCDLINE		equ	10000000B	; DCD and mask for modem line port

TXERRORMASK	equ	10000000B	; Transmit error mask (with UART reg.)
					; Bit 7 = timeout

RXERRORMASK	equ	10011110B	; Receive error mask (with UART reg.)
					; Bit 1 = rx overrun error
					; Bit 2 = rx parity error
					; Bit 3 = rx framing error
					; Bit 4 = rx break error
					; Bit 7 = timeout

IBMINT		equ	14h	; IBM serial interrupt
INIT_CMD	equ	0	; Int 14h init command
TX_CMD		equ	1	; Int 14h transmit command
RX_CMD		equ	2	; Int 14h receive command
STATUS_CMD	equ	3	; Int 14h status command



CODE	segment	byte	public	'CODE'

;********************** DATA STARTS HERE

segBASIC	dw	?	; MS-BASIC data segmant

pRtv		dw	?	; Pointers to basic variables
pDev		dw	?	; (relative to segBASIC)
pCom		dw	?
pDat		dw	?

Rtv		dw	?	; Local copies of basic variables
Dev		dw	?
Com		dw	?
Dat		dw	?

; The following are all SIO setup parameters

TxBaud		dw	8	; baud rate - 1200 baud default
TxDataBits	dw	8	; data bits - default 8
StopBits	dw	1	; stop bits - default 1
Parity		dw	0	; parity    - default none

; The following table is indexed by TxBaud and contains the initialisation byte
; OR values for the baud rate

BaudTable	label	byte
		db	0		; 0  = Illegal
		db	00000000B	; 1  = 110 baud
		db	00000000B	; 2  = 110
		db	00000000B	; 3  = 110
		db	00000000B	; 4  = 110
		db	00100000B	; 5  = 150
		db	01000000B	; 6  = 300
		db	01100000B	; 7  = 600
		db	10000000B	; 8  = 1200
		db	10000000B	; 9  = 1200
		db	10100000B	; A  = 2400
		db	10100000B	; B  = 2400
		db	11000000B	; C  = 4800
		db	11000000B	; D  = 4800
		db	11100000B	; E  = 9600
		db	11100000B	; F  = 9600

; The following table is indexed by Parity and contains the initialisation byte
; OR values for the parity

ParityTable	label	byte
		db	00000000B	; 0  = no parity
		db	00001000B	; 1  = odd
		db	00011000B	; 2  = even

; The following table is indexed by StopBits and contains the initialisation
; byte OR values for the number of stop bits

StopBitsTable	label	byte
		db	0		; 0  = Illegal
		db	00000000B	; 1  = 1 stop bit
		db	00000000B	; 2  = 1 stop bit
		db	00000100B	; 3  = 2 stop bits

; The following table is indexed by TxDataBits and contains the initialisation
; byte OR values for the number of data bits

DataBitsTable	label	byte
		db	0		; 0  = Illegal
		db	0		; 1  = Illegal
		db	0		; 2  = Illegal
		db	0		; 3  = Illegal
		db	0		; 4  = Illegal
		db	00000010B	; 5  = 7 data bits
		db	00000010B	; 6  = 7 data bits
		db	00000010B	; 7  = 7 data bits
		db	00000011B	; 8  = 8 data bits

; The jumptable follows with the offset of each routine

MAXCOMMAND	equ	12h

JumpTable	label	word	
	dw	offset	S_Init		; Init driver		0
	dw	offset	S_TxC		; Transmit character	1
	dw	offset	S_RxC		; Receive character	2
	dw	offset	S_EnableSIO	; Enable SIO settings	3
	dw	offset	S_ConfTxBaud	; Tx baud rate		4
	dw	offset	S_ConfRxBaud	; Rx baud rate		5
	dw	offset	S_TxDataBits	; Tx data bits		6
	dw	offset	S_RxDataBits	; Rx data bits		7
	dw	offset	S_StopBits	; Stop bits		8
	dw	offset	S_Parity	; Parity		9
	dw	offset	S_NoRoutine	; XON/XOFF transmit	A
	dw	offset	S_NoRoutine	; XON/XOFF receive	B
	dw	offset	S_RTS		; RTS line control	C
	dw	offset	S_DTR		; DTR line control	D
	dw	offset	S_NoRoutine	; Tx SIO control	E
	dw	offset	S_NoRoutine	; Rx SIO control	F	
	dw	offset	S_CTS		; Inquire CTS		10
	dw	offset	S_DCD		; Inquire DCD		11
	dw	offset	S_DSR		; Inquire DSR		12

;********************** CODE STARTS HERE

PUBLIC	IOINT


int14	macro
	push	ds
	int	IBMINT
	pop	ds
endm

assume	cs:CODE			; Tell assembler only CS set up

IOINT	proc	far

	push	bp
	mov	bp, sp

;****************************************************************
;* STACK AT THIS STAGE :-					*
;****************************************************************
;
;    SS:BP+0 	-> Saved BP
;	BP+2	-> Return offset
;	BP+4	-> Return segment
;	BP+6	-> Ret%
;	BP+8	-> Dat%
;	BP+10	-> Com%
;	BP+12	-> Dev%
;

	push	ax		; Save the world according to the great
	push	bx		; St. Basic
	push	cx
	push	dx
	push	si
	push	di
	push	ds
	push	es

	mov	ax, ds
	mov	cs:segBASIC, ax
	mov	es, ax		; ES now points to the basic data segment
	mov	ax, cs
	mov	ds, ax		; DS now points to my data/code segment

assume	cs:CODE, ds:CODE, es:NOTHING	; Tell the assembler this

	mov	si, 6[bp]	; Get the basic variables and their values
	mov	pRtv, si
	mov	ax, es:[si]
	mov	Rtv, ax

	mov	si, 8[bp]
	mov	pDat, si
	mov	ax, es:[si]
	mov	Dat, ax

	mov	si, 10[bp]
	mov	pCom, si
	mov	ax, es:[si]
	mov	Com, ax

	mov	si, 12[bp]
	mov	pDev, si
	mov	ax, es:[si]
	mov	Dev, ax

	cmp	ax, SERIALDEV	; Is this a request for the serial port handler?
	jne	IO_Finish	; Jump if not - its for some other device

	mov	si, Com		; Get command number
	cmp	si, MAXCOMMAND
	ja	IO_Finish	; Command beyond end of jump table
	add	si, si		; Word pointer
	call	JumpTable[si]

	mov	ax, segBASIC	; Just in case
	mov	es, ax

	mov	ax, Rtv		; Set the appropriate basic variables
	mov	si, pRtv
	mov	es:[si], ax

	mov	ax, Dat
	mov	si, pDat
	mov	es:[si], ax

IO_Finish:

	pop	es		; Restore the world
	pop	ds
	pop	di
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	bp

	ret	8		; Back to basic

IOINT	endp



;****************************************************************
;* Serial Procedures						*
;*								*
;* Here starts the serial port call handlers			*
;* All the following routines assume:				*
;*	Dev, Com, Dat and Rtv have been set up			*
;*	ds -> CODE						*
;****************************************************************



;****************************************************************
;* S_NoRoutine							*
;*								*
;* Dummy routine for calls not supported in this interface	*
;****************************************************************
S_NoRoutine	proc	near

	ret

S_NoRoutine	endp



;****************************************************************
;* S_Init							*
;*								*
;* Driver initialise routine                               	*
;****************************************************************
S_Init	proc	near

	mov	Rtv, 0			; Initialised OK
	ret

S_Init	endp



;****************************************************************
;* S_TxC							*
;*								*
;* Transmit character                                      	*
;*								*
;* Input parameters:						*
;*	Dat = character to be transmitted			*
;*								*
;* Output parameters:						*
;*	Rtv = 0      	Transmit completed OK			*
;*	      FFFFh(-1)	An error occurred			*
;*								*
;****************************************************************
S_TxC	proc	near

	mov	ax, Dat			; al = data
	mov	ah, TX_CMD
	mov	dx, PORTNUM		; dx = port
	int14
					; ah is now UART status register
	mov	dx, 0FFFFh		; dx is to be return value
	and	ah, TXERRORMASK
	or	ah, ah
	jnz	STC_Error
	xor	dx, dx			; No error = zero dx
STC_Error:
	mov	Rtv, dx
	ret

S_TxC	endp



;****************************************************************
;* S_RxC							*
;*								*
;* Receive character                                      	*
;*								*
;* Input parameters:						*
;*	None                              			*
;*								*
;* Output parameters:						*
;*	Rtv = character	   Receive completed OK			*
;*	      FFFFh(-1)	   An error occurred or no character	*
;*								*
;****************************************************************
S_RxC	proc	near

	INT	3
	mov	ah, RX_CMD
	mov	dx, PORTNUM		; dx = port
	int14
					; ah is now UART status register
	mov	dx, 0FFFFh		; dx is to be return value
	and	ah, RXERRORMASK
	or	ah, ah
	jnz	SRC_Error
	xor	dh, dh			; No error = zero dh
	mov	dl, al			; character received
SRC_Error:
	mov	Rtv, dx
	ret

S_RxC	endp



;****************************************************************
;* S_EnableSIO							*
;*								*
;* Enables all SIO setup calls 4 through 9                 	*
;*								*
;* Input parameters:						*
;*	None							*
;*								*
;* Output parameters:						*
;*	Rtv = 0      Completed OK              			* 
;*								*
;****************************************************************
S_EnableSIO	proc	near

	xor	al, al			; al is to become the initiasation byte
	mov	si, TxBaud		; get the value
	or	al, BaudTable[si]	; get the OR byte
	mov	si, Parity
	or	al, ParityTable[si]
	mov	si, StopBits
	or	al, StopBitsTable[si]
	mov	si, TxDataBits
	or	al, DataBitsTable[si]
	mov	dx, PORTNUM
	mov	ah, INIT_CMD		; initialise port
	int14
	mov	Rtv, 0
	ret

S_EnableSIO	endp



;****************************************************************
;* S_ConfTxBaud							*
;*								*
;* Configures transmit baud rate                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set (see table below)			*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting (see table below)			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;*	Value	Baud Rate	Value	Baud Rate		*							*
;*	1	N/A(110)      	9	N/A(1200)		*
;*	2	N/A(110)	A	2400			*
;*	3	110		B	N/A(2400)		*
;*	4	N/A(110) 	C	4800			*
;*	5	150		D	N/A(4800)		*
;*	6	300		E	9600            	*
;*	7	600            	F	N/A(19200)       	*
;*	8	1200						*
;*								*
;*	NOTE :- Certain Apricot baud rates are not supported	*
;*		by the IBM int 14h interface. These are		*
;*		marked by N/A in the table above and are	*
;*		replaced by the values shown			*
;*								*
;****************************************************************
S_ConfTxBaud	proc	near

	mov	ax, TxBaud
	mov	Rtv, ax			; Return current setting
	mov	ax, Dat
	cmp	ax, 1
	jb	SCTB_Illegal
	cmp	ax, 0Fh
	ja	SCTB_Illegal
	mov	TxBaud, ax		; Set to new value
SCTB_Illegal:
	ret

S_ConfTxBaud	endp



;****************************************************************
;* S_ConfRxBaud							*
;*								*
;* Configures receive baud rate                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set (see table in S_ConfTxBaud)	*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting (see table)			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;****************************************************************
S_ConfRxBaud	proc	near

	call	S_ConfTxBaud
	ret

S_ConfRxBaud	endp



;****************************************************************
;* S_TxDataBits							*
;*								*
;* Configures transmit data bits                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;*	NOTE :- If 5 or 6 data bits are requested, 7 data   	*
;*		bits will be used as only 7 or 8 data bits	*
;*		are supported by the IBM int 14h interface	*
;*								*
;****************************************************************
S_TxDataBits	proc	near

	mov	ax, TxDataBits
	mov	Rtv, ax			; Set return value
	mov	ax, Dat
	cmp	ax, 8
	ja	STDB_Illegal
	cmp	ax, 5
	jb	STDB_Illegal
	cmp	ax, 7
	jae	STDB_SetValue
	mov	ax, 7
STDB_SetValue:
	mov	TxDataBits, ax
STDB_Illegal:
	ret

S_TxDataBits	endp



;****************************************************************
;* S_RxDataBits							*
;*								*
;* Configures receive data bits                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;*	NOTE :- If 5 or 6 data bits are requested, 7 data   	*
;*		bits will be used as only 7 or 8 data bits	*
;*		are supported by the IBM int 14h interface	*
;*								*
;****************************************************************
S_RxDataBits	proc	near

	call	S_TxDataBits
	ret

S_RxDataBits	endp



;****************************************************************
;* S_StopBits							*
;*								*
;* Configures number of stop bits                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*	       1 = 1 stop bit					*
;*	       2 = 1 stop bit (1.5 not supported)		*
;*	       3 = 2 stop bits					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;****************************************************************
S_StopBits	proc	near

	mov	ax, StopBits
	mov	Rtv, ax			; Set return value
	mov	ax, Dat
	cmp	ax, 3
	ja	SSB_Illegal
	cmp	ax, 1
	jb	SSB_Illegal
	cmp	ax, 2
	je	SSB_SetValue
	mov	ax, 1			; 1.5 stop bits not supported
SSB_SetValue:
	mov	StopBits, ax
SSB_Illegal:
	ret

S_StopBits	endp



;****************************************************************
;* S_Parity							*
;*								*
;* Configures parity		                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*	       0 = No parity 					*
;*	       1 = Odd                           		*
;*	       2 = Even       					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*	      (Not operative until S_EnableSIO is called)	*
;*								*
;****************************************************************
S_Parity	proc	near

	mov	ax, Parity
	mov	Rtv, ax			; Set return value
	mov	ax, Dat
	cmp	ax, 2
	ja	SP_Illegal
	mov	Parity, ax
SP_Illegal:
	ret

S_Parity	endp



;****************************************************************
;* S_RTS   							*
;*								*
;* Sets or resets RTS		                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*	       0 = Reset line					*
;*	       1 = Set line                      		*
;*								*
;* Output parameters:						*
;*	Rtv = Present setting                  			* 
;*								*
;****************************************************************
S_RTS	proc	near

	mov	ax, Dat
	xor	ah, ah			; ah = 0
	or	ax, ax
	jz	SRTS_ResetLine		; jump if we wish to set RTS=0
	mov	ah, RTSLINE		; OR mask for RTS
SRTS_ResetLine:
	call	SetPortStatus
	xor	ah, ah			; ah
	shr	al, 1			; shift to bit 0
	mov	Rtv, ax			; return current setting
	ret

S_RTS	endp



;****************************************************************
;* S_DTR   							*
;*								*
;* Sets or resets DTR		                           	*
;*								*
;* Input parameters:						*
;*	Dat = Value to be set					*
;*	       0 = Reset line					*
;*	       1 = Set line                      		*
;*								*
;* Output parameters:						*
;*	Rtv = Present setting                  			* 
;*								*
;****************************************************************
S_DTR	proc	near

	mov	ax, Dat
	xor	ah, ah			; ah = 0
	or	ax, ax
	jz	SDTR_ResetLine		; jump if we wish to set DTR=0
	mov	ah, DTRLINE		; OR mask for DTR
SDTR_ResetLine:
	call	SetPortStatus
	xor	ah, ah			; ah = 0
	mov	Rtv, ax			; return current setting
	ret

S_DTR	endp



;****************************************************************
;* S_CTS							*
;*								*
;* returns status of clear to send                           	*
;*								*
;* Input parameters:						*
;*	None                  					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*		0 = Reset					*
;*		1 = Set						*
;*								*
;****************************************************************
S_CTS	proc	near

	call	GetPortStatus
	and	al, CTSLINE
	xor	dx, dx			; dx is to become return value
	or	al, al
	jz	SCTS_ResetLine
	inc	dx			; line is set - dx = 1
SCTS_ResetLine:
	mov	Rtv, dx		
	ret

S_CTS	endp



;****************************************************************
;* S_DCD							*
;*								*
;* returns status of data carrier detect                       	*
;*								*
;* Input parameters:						*
;*	None                  					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*		0 = Reset					*
;*		1 = Set						*
;*								*
;****************************************************************
S_DCD	proc	near

	call	GetPortStatus
	and	al, DCDLINE
	xor	dx, dx			; dx is to become return value
	or	al, al
	jz	SDCD_ResetLine
	inc	dx			; line is set - dx = 1
SDCD_ResetLine:
	mov	Rtv, dx		
	ret

S_DCD	endp



;****************************************************************
;* S_DSR							*
;*								*
;* returns status of data set ready                           	*
;*								*
;* Input parameters:						*
;*	None                  					*
;*								*
;* Output parameters:						*
;*	Rtv = Current setting                  			* 
;*		0 = Reset					*
;*		1 = Set						*
;*								*
;****************************************************************
S_DSR	proc	near

	call	GetPortStatus
	and	al, DSRLINE
	xor	dx, dx			; dx is to become return value
	or	al, al
	jz	SDSR_ResetLine
	inc	dx			; line is set - dx = 1
SDSR_ResetLine:
	mov	Rtv, dx		
	ret

S_DSR	endp



;****************************************************************
;* GetPortStatus						*
;*								*
;* Gets the serial port status bytes                           	*
;*								*
;* Input parameters:						*
;*	None                  					*
;*								*
;* Output parameters:						*
;*	ah  = UART status                      			* 
;*	al  = Modem line status                           	*
;*								*
;****************************************************************
GetPortStatus	proc	near

	mov	ah, STATUS_CMD
	mov	dx, PORTNUM
	int14
	ret

GetPortStatus	endp



;****************************************************************
;* SetPortStatus						*
;*								*
;* Sets the serial port status                           	*
;*								*
;* Input parameters:						*
;*	ah  = OR mask of bits to set (bits 0=DTR 1=RTS)		* 
;*								*
;* Output parameters:						*
;*	al  = Modem control port before setting			*
;*								*
;****************************************************************
SetPortStatus	proc	near

	push	bx
	push	dx
	mov	dx, MODEMCPORT
	in	al, dx			; get current port value
	jmp	short	sps1		; pause for back to back I/O
sps1:	mov	bl, al
	or	al, ah
	out	dx, al			; Set port value
	mov	al, bl			; restore al value
	pop	dx
	pop	bx
	ret

SetPortStatus	endp



Code	ends
end
