program Plotter;
{ demonstrates simple graphics techniques }

var
  Dev, Com, Dat, Ret : integer;      { used by Apricot Control Device }
  PixelType : integer;
  LineType  : integer;
  XCurrent, YCurrent : integer;
  XMin, YMin, XMax, YMax : integer;  { window size for clipping }
  DefSeg : integer;

type
  String6   = string[6];

{$V-}

procedure Initialise;
{ initialise variables }
begin
  DefSeg := 640;
  PixelType := 1;   { pixels will be 'on' }
  LineType  := 1;   { use simple line drawing algorithm }
  XMin := 0; YMin := 0; XMax := 799; YMax := 399;  { full screen window }
end;

function FnIntPower(X,Y: integer): integer;
{ returns X to power of Y for Y >= 0 }
{ could use shl function, but seems to run as fast using this technique }
label 1;
var
  J,K: integer;
begin
  K := 1;
  if Y = 0 then goto 1;
  for J := 1 to Y do K := K * X;
  1: FnIntPower := K;
end;

procedure ControlDevice;
{ Apricot Control Device }
type
  Result = record
             AX,BX,CX,DX,BP,SI,DI,DS,ES,Flags: integer;
           end;
var
  Device : Result;
begin
  with Device do
  begin
    BX := Dev;
    CX := Com;
    DX := Dat;
  end;
  intr($0FC,Device);
  with Device do
  begin
    Dat := DX;
    Ret := AX;
  end;
end;

procedure TextMode;
{ text mode }
begin
  Dev:=49; Com:=1; Dat:=0; ControlDevice;
end;

procedure GraphMode;
{ graphics mode }
var
  DSeg, Address, Pointer, I : integer;
begin
  Dev:=49; Com:=1; Dat:=1; ControlDevice;
  { Now set screen RAM pointers }
  DSeg := 30000 + 31440;  { 61440, $F000 }
  Address := 0;
  Pointer := 320;  { 640/2 }
  for I := 0 to 1249 do
  begin
    memw[DSeg: Address] := Pointer;
    Address := Address + 2;
    Pointer := Pointer + 1;
  end;
end;

procedure Clear(Pattern : byte);
{ clears graphics screen if Pattern = 0 }
begin
  fillchar(memw[DefSeg: 0], 2*20000, Pattern);
end;

procedure Pixel(X, Y: integer);
{ plots a single pixel }
label 1;
var
  Address : integer;
  Row, Col, ScanLine, ScanPoint : integer;
begin
  if (X < XMin) or (X > XMax) or (Y < YMin) or (Y > YMax) then goto 1;
  Row := Y div 16;
  Col := X div 16;
  ScanLine := Y mod 16;
  ScanPoint := X mod 16;
  Address := (Row*50 + Col)*16 + ScanLine;
  { 'and not' to clear pixel }
  { 'or' to set pixel, 'xor' to change state of pixel }
  case PixelType of
    0: memw[DefSeg: 2*Address] := (memw[DefSeg: 2*Address] and not FnIntPower(2,ScanPoint));
    1: memw[DefSeg: 2*Address] := (memw[DefSeg: 2*Address] or FnIntPower(2,ScanPoint));
    2: memw[DefSeg: 2*Address] := (memw[DefSeg: 2*Address] xor FnIntPower(2,ScanPoint));
  end;
  XCurrent := X;
  YCurrent := Y;
  1:;
end;

{ PixelOff, PixelOn, PixelToggle }
{ these set type of pixel drawing }
{ 'off', 'on' and 'toggle' }
procedure PixelOff;    begin PixelType := 0; end;
procedure PixelOn;     begin PixelType := 1; end;
procedure PixelToggle; begin PixelType := 2; end;

procedure GraphWindow(X1, Y1, X2, Y2: integer);
{ sets graphics window for clipping }
begin
  if X1 < X2 then begin XMin := X1; XMax := X2; end;
  if X1 > X2 then begin XMin := X2; XMax := X1; end;
  if Y1 < Y2 then begin YMin := Y1; YMax := Y2; end;
  if Y1 > Y2 then begin YMin := Y2; YMax := Y1; end;
  if XMin < 0 then XMin := 0;
  if XMax > 799 then XMax := 799;
  if YMin < 0 then YMin := 0;
  if YMax > 399 then YMax := 399;
end;

procedure SimpleLine(X1, Y1, X2, Y2: integer);
{ from 'Principles of Interactive Computer Graphics' }
{ by William M Newman and Robert F Sproull, p24      }
{ 'simple DDA (digital differential analyser)'       }
var
  Length, I  : integer;
  X, Y       : real;
  XIncrement, YIncrement : real;
begin
  XCurrent := X2;
  YCurrent := Y2;
  Length := abs(X2 - X1);
  if abs(Y2 - Y1) > Length then Length := abs(Y2 - Y1);
  XIncrement := (X2 - X1)/Length;
  YIncrement := (Y2 - Y1)/Length;
  X := X1 + 0.5;
  Y := Y1 + 0.5;
  for I := 1 to Length do
  begin
    Pixel(trunc(X), trunc(Y));
    X := X + XIncrement;
    Y := Y + YIncrement;
  end;
  Pixel(X2,Y2);   { ensures that the last pixel is drawn }
end;

procedure TestLine(X1, Y1, X2, Y2: integer);
{ based on SimpleLine }
{ from 'Principles of Interactive Computer Graphics' }
{ by William M Newman and Robert F Sproull, p24      }
{ 'simple DDA (digital differential analyser)'       }
var
  Length, I  : integer;
  X, Y       : real;
  XIncrement, YIncrement : real;
begin
  XCurrent := X2;
  YCurrent := Y2;
  Length := abs(X2 - X1);
  if abs(Y2 - Y1) > Length then Length := abs(Y2 - Y1);
  XIncrement := (X2 - X1)/Length;
  YIncrement := (Y2 - Y1)/Length;
  X := X1 + 0.5;
  Y := Y1 + 0.5;
  for I := 1 to Length do
  begin
    Pixel(trunc(X), trunc(Y));
    X := X + XIncrement;
    Y := Y + YIncrement;
  end;
  Pixel(X2,Y2);   { ensures that the last pixel is drawn }
end;

procedure BresLine(X1, Y1, X2, Y2 : integer);
{ from 'Principles of Interactive Computer Graphics' }
{ by William M Newman and Robert F Sproull, p26      }
{ 'Bresenham's Algorithm'                            }
{ Note: not yet debugged }
{ DeltaX and DeltaY must be non-zero }
{ algorithm doubles line width if run in both directions }
var
  X, Y, E, I, DeltaX, DeltaY : integer;
begin
  XCurrent := X2;
  YCurrent := Y2;
  DeltaX := X2 - X1;
  DeltaY := Y2 - Y1;
  X := X1;
  Y := Y1;
  E := 2*DeltaY - DeltaX;
  if DeltaY >= 0 then
  begin
    if DeltaX >= 0 then
    begin
      for I := 1 to DeltaX do
      begin
        Pixel(X,Y);
        if E > 0 then
        begin
          Y := Y + 1;
          E := E + (2*DeltaY - 2*DeltaX);
        end
        else E := E + 2*DeltaY;
        X := X + 1;
      end;
    end
    else
    begin
      for I := 1 downto DeltaX do
      begin
        Pixel(X,Y);
        if E > 0 then
        begin
          Y := Y + 1;
          E := E + (2*DeltaY + 2*DeltaX);
        end
        else E := E + 2*DeltaY;
        X := X - 1;
      end;
    end;
  end
  else
  begin
    if DeltaX >= 0 then
    begin
      for I := 1 to DeltaX do
      begin
        Pixel(X,Y);
        if E > 0 then
        begin
          Y := Y - 1;
          E := E + (-2*DeltaY - 2*DeltaX);
        end
        else E := E - 2*DeltaY;
        X := X + 1;
      end;
    end
    else
    begin
      for I := 1 downto DeltaX do
      begin
        Pixel(X,Y);
        if E > 0 then
        begin
          Y := Y - 1;
          E := E + (-2*DeltaY + 2*DeltaX);
        end
        else E := E - 2*DeltaY;
        X := X - 1;
      end;
    end;
  end;
end;

procedure Line(X1, Y1, X2, Y2 : integer);
{ selects method of line drawing required }
begin
  case LineType of
    1: SimpleLine(X1, Y1, X2, Y2);
    2: BresLine(X1, Y1, X2, Y2);
    3: TestLine(X1, Y1, X2, Y2);
  end;
end;

procedure Hold;
{ waits for keystroke }
var
  X : char;
begin
  read(kbd,X);
end;

procedure LineTo(X, Y : integer);
{ draws line from last current point }
begin
  Line(XCurrent,YCurrent,X,Y);
end;

procedure MoveTo(X, Y : integer);
{ moves current point }
begin
  XCurrent := X;
  YCurrent := Y;
end;

procedure Box(X1, Y1, X2, Y2 : integer);
{ draws box }
begin
  Line(X1,Y1,X1,Y2);
  LineTo(X2,Y2);
  LineTo(X2,Y1);
  LineTo(X1,Y1);
end;

procedure SineCurve;
{ draws demo sine curve }
var
  X, Y : integer;
begin
  MoveTo(40,200);
  for X := 1 to 720 do
  begin
    Y := 200 - trunc(150*sin(X*2*pi/360));
    LineTo(40+X,Y);
  end;
end;

begin
  Initialise;
  GraphMode;
  Clear(0);
  Line(0,0,799,399);
  Line(0,399,799,0);
  Box(100,100,700,300);
  Box(0,0,799,399);
  SineCurve;
  Hold;
  TextMode;
end.
